(function(){

    const baseurl = "https://digitallibrary-external.fastretailing.com/";
    const fetchNumber = 30;
    const sleepTime = 2000;
    let allowDebugMessage = true;

    function addLightBox(){
        if(document.URL === baseurl){
            fetch(chrome.extension.getURL("fetch/lightbox.html"))
            .then(response => response.text())
            .then(result => document.body.insertAdjacentHTML('beforeend', result))
            .then(()=>{
                // turn on/ off lightbox
                document.querySelector("#lightbox-switch").addEventListener("click",function(){
                    document.getElementById("fetch-lightbox").classList.add("on");
                });
                document.querySelector("#fetch-lightbox .close-btn").addEventListener("click",function(){
                    document.getElementById("fetch-lightbox").classList.remove("on");
                });

                // operation
                document.querySelector("#saveSession").addEventListener("click",saveSession);
                document.querySelector("#loadSession").addEventListener("click",loadSession);

                document.querySelector("#searchSKUs").addEventListener("click", startFetching);
                document.querySelector("#downloadSelectedSKUs").addEventListener("click",downloadFetching);
                document.querySelector("#downloadAllSKUs").addEventListener("click",extensionDownloading);
                document.querySelector("#filter-date").addEventListener("change",filterByDate);
 
                document.querySelector("#fetch-lightbox .debug-box").addEventListener("click",function(event){
                    if(!event.target.classList.contains("debug-box")){
                        event.target.closest('.asset-item').classList.toggle("on");
                    }
                });
                
            })
            .catch(error => console.log('error', error));
        }
    }

    /* Utility function */

    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    function addDebugMessage(str){
        if(allowDebugMessage){
            let div = document.createElement("div");
            let content = document.createTextNode(str);
    
            div.appendChild(content);
            document.querySelector("#fetch-lightbox .debug-box").appendChild(div);
        }else{
            document.querySelector("#fetch-lightbox .info-message").innerHTML = str;
        }
    }

    function generateCookie(){
        let cookie = document.cookie;
        // HTTPOnly cookie require manual input
        cookie += "; digitallibrary-session=" + document.getElementById("session-string").value;
        cookie += "; digitallibrary-session.sig=" + document.getElementById("session-sig-string").value;
        return cookie;
    }

    function getHeader(){
        var header = new Headers();
        header.append("authority", "digitallibrary-external.fastretailing.com");
        header.append("pragma", "no-cache");
        header.append("cache-control", "no-cache");
        header.append("user-agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/96.0.4664.110 Safari/537.36");
        header.append("accept", "*/*");
        header.append("accept-language", "en-US,en;q=0.9,zh-TW;q=0.8,zh;q=0.7,zh-CN;q=0.6,ja;q=0.5");
        header.append("cookie", generateCookie());
        return header;
    }

    function getConfig(header){
        return {
            "method": "GET",
            "headers": header,
            "mode": "cors",
            "redirect": "follow"
        };
    }

    function saveSession(){
        let sessionString = document.getElementById("session-string").value;
        let sessionSigString = document.getElementById("session-sig-string").value;
        chrome.storage.local.set({
            "sessionString": sessionString,
            "sessionSigString": sessionSigString
        });
    }

    function loadSession(){
        chrome.storage.local.get(["sessionString", "sessionSigString"],function(result){
            document.getElementById("session-string").value = result.sessionString;
            document.getElementById("session-sig-string").value = result.sessionSigString
        });
    }


    /* End of utility function */

    /* Start of main function */

    // Reading the textarea and fetch the SKUs one by one
    // then create DOM and append it
    async function startFetching(){
        let SKUString = document.getElementById("sku-list").value;
        if(SKUString.length){
            let SKUlist = SKUString.split("\n");
            let container = document.querySelector("#fetch-lightbox .debug-box");
            allowDebugMessage = false;

            // remove container innerHTML
            while(container.hasChildNodes()){
                container.removeChild(container.firstChild);
            }
            document.querySelector("#fetch-lightbox .info-message").innerHTML = "";
            
            for(let SKU of SKUlist){
                let assetlist = await fetchSKU(SKU);
                
                // simple filtering
                assetlist = assetlist.filter(asset => asset.file_extension === "jpg" || asset.file_extension === "gif");
                assetlist = assetlist.filter(asset => !asset.filename.includes("_I0"));

                // create DOM string (same as innerHTML) then join it and append it
                let DOMString = assetlist.map(asset => {
                    return `
                        <div class="asset-item">
                            <div><img src="/api/asset/${asset.id}/thumbnail/medium?r=1" loading="lazy"/></div>
                            <p>
                                ${SKU}<br />
                                <span class="asset-name">${asset.filename}</span>.${asset.file_extension}
                            </p>
                            <span class="asset-id" hidden>${asset.id}</span>
                            <span class="asset-version" hidden>${asset.fileVersion}</span>
                            <span class="asset-date" hidden>${asset.updatedDate}</span>
                        </div>
                    `;
                }).join('');
                container.insertAdjacentHTML('beforeend', DOMString);

                // client side filteing
                filterByDate();
            }

            addDebugMessage(`SKU search completed.`);

        }else{
            alert("It is empty on SKU list.");
        }
    }

    // this function base on startFetching, will be used after user selected all assets
    async function downloadFetching(){
        // grab selected DOM, and turn from NodeList to Array
        let DOMArray = [...document.querySelectorAll("#fetch-lightbox .debug-box .asset-item.on")];
        
        // turn DOM into object array with id and file version
        let downloadlist = DOMArray.map(x => {
            return {
                "id" : x.querySelector(".asset-id").innerHTML,
                "fileVersion" :  x.querySelector(".asset-version").innerHTML
            };
        });

        if(downloadlist.length >= 2)
            await downloadList(downloadlist, `asset_file`);
        else if(downloadlist.length === 1)
            await downloadList(downloadlist, document.querySelector("#fetch-lightbox .debug-box .asset-item.on .asset-name").innerHTML);
        else
            alert("There is no selected asset.");

        // remove the clicked state on asset item
        document.querySelectorAll("#fetch-lightbox .debug-box .asset-item.on").forEach(x => x.classList.remove("on"));
    }

    // perform operation similar to old extension
    async function extensionDownloading(){
        let SKUString = document.getElementById("sku-list").value;
        if(SKUString.length){
            let SKUlist = SKUString.split("\n");
            let container = document.querySelector("#fetch-lightbox .debug-box");
            allowDebugMessage = true;

            while(container.hasChildNodes()){
                container.removeChild(container.firstChild);
            }
            document.querySelector("#fetch-lightbox .info-message").innerHTML = "";

            for(let SKU of SKUlist){
                let assetlist = await fetchSKU(SKU);
                let downloadlist = filtering(assetlist, SKU);
                await downloadList(downloadlist["product"], `${SKU}_product`);
                await sleep(sleepTime);
                await downloadList(downloadlist["model"], `${SKU}_model`);
                await sleep(sleepTime);
            }
        }else{
            alert("It is empty on SKU list.");
        }
    }

    //grabbing single SKU data
    async function fetchSKU(SKU = ""){
        if(SKU.length == 0){
            return;
        }

        let currentms = new Date().getTime();
        let url = `${baseurl}api/asset?time=${currentms}&t=datacategory%2Fasset&w=${SKU}&r=north&z=${fetchNumber}&c=DEFAULT`;
        let header = getHeader();
        let config = getConfig(header);

        addDebugMessage(`Start fetching SKU=${SKU}`);

        return await fetch(url,config)
        .then(response => response.json())
        .then(async function(result){

            await sleep(sleepTime);

            //get json data
            let assetlist = result.assets;
            let scrollId = result.scrollId;
            let total = result.total;

            // fetch the remaining item by simluating scroll operation
            for(let i = total - fetchNumber ; i > 0 ; i -= fetchNumber){
                addDebugMessage(`fetching SKU=${SKU} Process: ${i} remains`);

                let response = await fetch(`${baseurl}api/asset/more?scrollId=${scrollId}`, config)
                .then(response => response.json())
                .catch(error => addDebugMessage(error));

                assetlist = assetlist.concat(response.assets);

                await sleep(sleepTime);
            }
            addDebugMessage(`fetching SKU=${SKU} completed, total item: ${total}`);

            return assetlist;
        })
        .catch(error => console.error(error));
    }

    // filter asset by Date
    // only used on Search and Download
    function filterByDate(){
        let datefilter = 0;
        if(document.getElementById("filter-date").value){
            datefilter = new Date().setHours(0,0,0,0) / 1000 - document.getElementById("filter-date").value * 86400;
        }
        document.querySelectorAll("#fetch-lightbox .debug-box .asset-item").forEach(asset => {
            asset.classList.remove("on");
            if(asset.querySelector(".asset-date").innerHTML > datefilter){
                asset.style.display = "";
            }else{
                asset.style.display = "none";
            }
        });
    }

    // filter asset to different array
    // only used on Download All SKUs
    // usage:
    //     let downloadlist = filtering(assetlist, SKU);

    function filtering(assetlist, SKU){
        assetlist = assetlist.filter(asset => asset.file_extension === "jpg" || asset.file_extension === "gif");

        let multilist = assetlist.reduce(function(arrlist, item){

            if( item.filename.includes(`goods_${SKU}_sub`) || 
                item.filename.includes(`_${SKU}_chip`) || 
                (item.filename.includes(SKU) && item.models.length === 0 && item.file.filesize <= 1048576) ){  // file name include SKU, no model and size <= 1MB
                arrlist["product"].push(item);

            }else if( (item.filename.includes(SKU) && item.models.length > 0) || (item.filename.includes("_M0"))){
                arrlist["model"].push(item);
            }
            return arrlist;

        }, {"product" : [], "model" : []}); // initial value

        return multilist;
    }

    // given the asset id and file version to download the asset
    // downloadList usage:
    //    await downloadList(downloadlist["product"], "123456_product");
    //    await downloadList(downloadlist["model"], "123456_model");
    
    async function downloadList(assetlist, name){
        if(assetlist.length >= 2){

            let id = assetlist.map(asset => asset.id);
            let revision = assetlist.map(asset => asset.fileVersion);

            let json = JSON.stringify({id : id, revision: revision});
            console.log(json);

            // prepare header for POST request
            let postheader = getHeader();
            postheader.append("Content-Type", "application/json");
            postheader.append("csrf-token", document.querySelector('meta[name="csrf-token"]').content);
            postheader.append("origin", "https://digitallibrary-external.fastretailing.com");

            // send POST request for getting download token
            let response = await fetch(`${baseurl}api/asset/downloadtoken`,{
                "method": "POST",
                "headers": postheader,
                "mode": "cors",
                "redirect": "follow",
                "body": json
            })
            .then(response => response.json())
            .catch(error => console.error(error));

            // send GET request for downloading the zip file
            fetch(`/api/asset/download/${response.token}`, getConfig(getHeader()))
            .then(response => response.blob())
            .then(blob => {

                // https://stackoverflow.com/questions/32545632/how-can-i-download-a-file-using-window-fetch
                let url = window.URL.createObjectURL(blob);
                let a = document.createElement('a');
                a.href = url;
                a.download = `${name}.zip`;
                document.body.appendChild(a); // we need to append the element to the dom -> otherwise it will not work in firefox
                a.click();    
                a.remove();  //afterwards we remove the element again     
                
                addDebugMessage(`download ${name}.zip completed`);
            })
            .catch(error => console.error(error));
        
        }else if(assetlist.length === 1){

            // just use anchor tag to download single file
            let id = assetlist[0].id;
            let filename = assetlist[0].filename;
            if(filename === undefined)
                filename = name;
            let downloadDOM = document.getElementById("download-trigger");
            downloadDOM.setAttribute('href', `/api/asset/${id}/download/`);
            downloadDOM.setAttribute('download', filename);
            downloadDOM.click();

            addDebugMessage(`download file ${filename} completed`);
        }else{
            addDebugMessage(`asset list is empty.`);
        }
    }

    addLightBox();

})();