(function () {

    const baseurl = "https://digitallibrary-external.fastretailing.com/";
    const fetchNumber = 30;
    const sleepTime = 2500;
    let allowDebugMessage = true;

    // adding the lightbox (control panel) to the body
    // by grabbing the lightbox.html
    async function addLightBox() {
        if (document.URL === baseurl) {
            try{

                let response = await fetch(chrome.extension.getURL("fetch/lightbox.html"));
                let result = await response.text();
                document.body.insertAdjacentHTML('beforeend', result);

                // turn on/ off lightbox
                document.querySelector("#lightbox-switch").addEventListener("click", function () {
                    document.getElementById("fetch-lightbox").classList.add("on");
                });
                document.querySelector("#fetch-lightbox .close-btn").addEventListener("click", function () {
                    document.getElementById("fetch-lightbox").classList.remove("on");
                });

                document.querySelector("#searchSKUs").addEventListener("click", startFetching);
                document.querySelector("#downloadSelectedSKUs").addEventListener("click", downloadFetching);
                document.querySelector("#downloadAllSKUs").addEventListener("click", extensionDownloading);

                document.querySelectorAll("#filter-date, #show-product-box, #show-model-box, #filter-unrelated").forEach(function(item){
                    item.addEventListener("change", clientSideFilter);
                });

                document.querySelector("#fetch-lightbox .debug-box").addEventListener("click", function (event) {
                    if (!event.target.classList.contains("debug-box")) {
                        event.target.closest('.asset-item').classList.toggle("on");
                    }
                });

            }catch(error){
                console.error("Unable to fetch lightbox.html" + error);
            }
        }
    }

    /* Utility function */

    function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    function addDebugMessage(str) {
        if (allowDebugMessage) {
            let div = document.createElement("div");
            let content = document.createTextNode(str);

            div.appendChild(content);
            document.querySelector("#fetch-lightbox .debug-box").appendChild(div);
        } else {
            document.querySelector("#fetch-lightbox .info-message").innerHTML = str;
        }
    }

    function getConfig() {
        return {
            "method": "GET",
            "mode": "cors",
            "redirect": "follow"
        };
    }

    /* End of utility function */

    /* Start of main function */

    // Reading the textarea and fetch the SKUs one by one
    // then create DOM and append it
    async function startFetching() {
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = true);
        let SKUString = document.getElementById("sku-list").value;
        if (SKUString.length) {
            let SKUlist = SKUString.split("\n");
            let container = document.querySelector("#fetch-lightbox .debug-box");
            allowDebugMessage = false;

            // remove container innerHTML
            while (container.hasChildNodes()) {
                container.removeChild(container.firstChild);
            }
            document.querySelector("#fetch-lightbox .info-message").innerHTML = "";

            for (let SKU of SKUlist) {

                // fetch product URL
                let currentms = new Date().getTime();
                let assetlist = await fetchSKU(`${baseurl}api/asset?time=${currentms}&t=main_product%2FGU%2F${SKU}&t=datacategory%2Fecasset&r=north&z=${fetchNumber}&c=DEFAULT`, `${SKU} product image`);
                let DOMString = createDOMString(SKU, assetlist); 
                container.insertAdjacentHTML('beforeend', DOMString);
                clientSideFilter();

                // fetch model URL
                currentms = new Date().getTime();
                assetlist = await fetchSKU(`${baseurl}api/asset?time=${currentms}&t=datacategory%2Fasset&w=${SKU}&r=north&z=${fetchNumber}&c=DEFAULT`, `${SKU} model image`);
                DOMString = createDOMString(SKU, assetlist); 
                container.insertAdjacentHTML('beforeend', DOMString);
                clientSideFilter();
            }

            addDebugMessage(`SKU search completed.`);
        } else {
            alert("It is empty on SKU list.");
        }
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = false);
    }

    // this function base on startFetching, will be used after user selected all assets
    async function downloadFetching() {
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = true);
        // grab selected DOM, and turn from NodeList to Array
        let DOMArray = [...document.querySelectorAll("#fetch-lightbox .debug-box .asset-item.on")];

        // turn DOM into object array with id and file version
        let downloadlist = DOMArray.map(x => {
            return {
                "id": x.querySelector(".asset-id").innerHTML,
                "fileVersion": x.querySelector(".asset-version").innerHTML
            };
        });

        if (downloadlist.length >= 2)
            await downloadList(downloadlist, `asset_file`);
        else if (downloadlist.length === 1)
            await downloadList(downloadlist, document.querySelector("#fetch-lightbox .debug-box .asset-item.on .asset-name").innerHTML);
        else
            alert("There is no selected asset.");

        // remove the clicked state on asset item
        document.querySelectorAll("#fetch-lightbox .debug-box .asset-item.on").forEach(x => x.classList.remove("on"));
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = false);
    }

    function createDOMString(SKU, assetlist){
        
        let multilist = filtering(assetlist, SKU);

        let createString = (asset, SKU, classes) => {
            return `
                <div class="asset-item ${classes}">
                    <div><img src="/api/asset/${asset.id}/thumbnail/medium?r=1" loading="lazy"/></div>
                    <p>
                        <span class="asset-SKU">${SKU}</span><br />
                        <span class="asset-name">${asset.filename}</span>.${asset.file_extension}
                    </p>
                    <span class="asset-id" hidden>${asset.id}</span>
                    <span class="asset-version" hidden>${asset.fileVersion}</span>
                    <span class="asset-date" hidden>${asset.updatedDate}</span>
                </div>
            `;
        };

        let productString = multilist["product"].map(x => createString(x, SKU, "is-product")).join("");
        let modelString = multilist["model"].map(x => createString(x, SKU, "is-model")).join("");
        let unrelatedString = multilist["unrelated"].map(x => createString(x, SKU, "is-unrelated")).join("");

        return productString + modelString + unrelatedString;
    }

    // perform operation similar to old extension
    async function extensionDownloading() {
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = true);
        let SKUString = document.getElementById("sku-list").value;
        if (SKUString.length) {
            let SKUlist = SKUString.split("\n");
            let container = document.querySelector("#fetch-lightbox .debug-box");
            allowDebugMessage = true;

            while (container.hasChildNodes()) {
                container.removeChild(container.firstChild);
            }
            document.querySelector("#fetch-lightbox .info-message").innerHTML = "";

            for (let SKU of SKUlist) {

                // first link
                let currentms = new Date().getTime();
                let assetlist = await fetchSKU(`${baseurl}api/asset?time=${currentms}&t=main_product%2FGU%2F${SKU}&t=datacategory%2Fecasset&r=north&z=${fetchNumber}&c=DEFAULT`, `${SKU} product image`);
                let downloadlist = filtering(assetlist, SKU);
                await downloadList(downloadlist["product"], `${SKU}`);
                await sleep(sleepTime);

                let is_old = assetlist.length === 0;
                let modellist = downloadlist["model"];

                // second link
                currentms = new Date().getTime();
                assetlist = await fetchSKU(`${baseurl}api/asset?time=${currentms}&t=datacategory%2Fasset&w=${SKU}&r=north&z=${fetchNumber}&c=DEFAULT`, `${SKU} model image`);
                downloadlist = filtering(assetlist, SKU);

                if(is_old){
                    addDebugMessage(`re-attempt download product list`);
                    await downloadList(downloadlist["product"], `${SKU}`);
                    await sleep(sleepTime);
                }

                await downloadList(downloadlist["model"].concat(modellist), `${SKU}_model`);
                await sleep(sleepTime);
            }
        } else {
            alert("It is empty on SKU list.");
        }
        document.querySelectorAll("#fetch-lightbox button").forEach(x => x.disabled = false);
    }

    //grabbing single SKU data
    async function fetchSKU(url = "", message = "") {
        if (message.length == 0 || url.length == 0) {
            return;
        }
        
        let assetlist = [];
        let config = getConfig();
        addDebugMessage(`Start fetching SKU=${message}`);

        try{
            let request = await fetch(url, config);
            let response = await request.json();

            await sleep(sleepTime);

            assetlist = response.assets;
            let scrollId = response.scrollId;
            let total = response.total;

            // fetch the remaining item by simluating scroll operation
            for (let i = total - fetchNumber; i > 0; i -= fetchNumber) {
                addDebugMessage(`fetching SKU=${message} Process: ${i} remains`);

                let req = await fetch(`${baseurl}api/asset/more?scrollId=${scrollId}`, config);
                let res = await req.json();
                assetlist = assetlist.concat(res.assets);

                await sleep(sleepTime / 1.5);
            }
            addDebugMessage(`fetching SKU=${message} completed, total item: ${total}`);

        }catch(error){
            addDebugMessage("Fetch error on fetchSKU : " + error);
            throw new Error("Fetch error on fetchSKU : " + error);
        }

        return assetlist;
    }

    // filter asset by Date
    // only used on Search and Download
    function clientSideFilter() {
        let datefilter = 0;
        let checkProduct = !document.getElementById("show-product-box").checked;
        let checkModel = !document.getElementById("show-model-box").checked;
        let checkRelated = document.getElementById("filter-unrelated").checked;

        if (document.getElementById("filter-date").value) {
            datefilter = new Date().setHours(0, 0, 0, 0) / 1000 - document.getElementById("filter-date").value * 86400;
        }
        document.querySelectorAll("#fetch-lightbox .debug-box .asset-item").forEach(asset => {
            asset.classList.remove("on");
            asset.style.display = "none";

            if (asset.querySelector(".asset-date").innerHTML > datefilter &&
                (!checkProduct || (checkProduct && !asset.classList.contains("is-product"))) &&
                (!checkModel || (checkModel && !asset.classList.contains("is-model"))) &&
                (!checkRelated || (checkRelated && !asset.classList.contains("is-unrelated")))
            ) {
                asset.style.display = "";
            }
        });
    }

    // filter asset to different array
    // usage:
    //     let downloadlist = filtering(assetlist, SKU);

    function filtering(assetlist, SKU) {
        assetlist = assetlist.filter(asset => asset.file_extension === "jpg" || asset.file_extension === "gif");

        let multilist = assetlist.reduce(function (arrlist, item) {
            
            let tagid = item.tags.map(x => x.id);

            if ( item.filename.match(new RegExp(`^goods_${SKU}_sub`))  ||
                 item.filename.includes(`_${SKU}_chip`) ||
                (item.filename.match(new RegExp(`^goods_[0-9]{2}_${SKU}$`)) && item.models.length === 0 && item.products.length === 1) // file name include SKU and no model
            ) {  
                arrlist["product"].push(item);

            } else if (item.filename.includes(`${SKU}_I0`) ||
                (item.filename.includes("goods") && ( !item.filename.includes(SKU) || item.filename[0] !== 'h' )) ||
                tagid.includes("assettype/Poster") || 
                item.filename.includes("GU_") || 
                item.filename.match(new RegExp(`^[0-9]{6}_.{8}_[0-9]{2}_.{4}`))
            ) {
                arrlist["unrelated"].push(item);

            }else if ( tagid.find(x => x.includes("Model") || x.includes("Appeal POP")) ) {
                arrlist["model"].push(item);
            }else{
                arrlist["unrelated"].push(item);
            }
            return arrlist;

        }, { "product": [], "model": [], "unrelated": [] }); // initial value

        return multilist;
    }

    // given the asset id and file version to download the asset
    // downloadList usage:
    //    await downloadList(downloadlist["product"], "123456_product");
    //    await downloadList(downloadlist["model"], "123456_model");

    async function downloadList(assetlist, name) {
        if (assetlist.length >= 2) {

            let id = assetlist.map(asset => asset.id);
            let revision = assetlist.map(asset => asset.fileVersion);

            let json = JSON.stringify({ id: id, revision: revision });
            console.log(json);

            // prepare header for POST request
            let postheader = new Headers();
            postheader.append("Content-Type", "application/json");
            postheader.append("csrf-token", document.querySelector('meta[name="csrf-token"]').content);
            postheader.append("origin", "https://digitallibrary-external.fastretailing.com");

            let response;
            // send POST request for getting download token
            try{
                let request = await fetch(`${baseurl}api/asset/downloadtoken`, {
                    "method": "POST",
                    "headers": postheader,
                    "mode": "cors",
                    "redirect": "follow",
                    "body": json
                
                });
                response = await request.json();

            }catch(error){
                addDebugMessage("Fetch error on DownloadList - POST request: " + error);
                throw new Error("Fetch error on DownloadList - POST request: " + error);
            }

            // send GET request for downloading the zip file
            if(response){
                try{
                    let request = await fetch(`/api/asset/download/${response.token}`, getConfig());
                    let blob = await request.blob();

                    // https://stackoverflow.com/questions/32545632/how-can-i-download-a-file-using-window-fetch
                    let url = window.URL.createObjectURL(blob);
                    let a = document.createElement('a');
                    a.href = url;
                    a.download = `${name}.zip`;
                    document.body.appendChild(a); // we need to append the element to the dom -> otherwise it will not work in firefox
                    a.click();
                    a.remove();  //afterwards we remove the element again     

                    addDebugMessage(`download ${name}.zip completed`);

                }catch(error){
                    addDebugMessage("Fetch error on DownloadList - GET request: " + error);
                    throw new Error("Fetch error on DownloadList - GET request: " + error);
                }
            }

        } else if (assetlist.length === 1) {

            // just use anchor tag to download single file
            let id = assetlist[0].id;
            let filename = assetlist[0].filename;
            if (filename === undefined)
                filename = name;
            let downloadDOM = document.getElementById("download-trigger");
            downloadDOM.setAttribute('href', `/api/asset/${id}/download/`);
            downloadDOM.setAttribute('download', filename);
            downloadDOM.click();

            addDebugMessage(`download file ${filename} completed`);
        } else {
            addDebugMessage(`asset list is empty.`);
        }
    }

    addLightBox();
})();